package com.cftech.logistics.web;

import com.alibaba.fastjson.JSONObject;
import com.cftech.logistics.model.Logistics;
import com.cftech.logistics.service.LogisticsService;
import com.cftech.core.poi.ExcelKit;
import com.cftech.core.scope.OrderType;
import com.cftech.core.sql.Conds;
import com.cftech.core.sql.Sort;
import com.cftech.core.util.Constants;
import com.cftech.sys.security.PermissionSign;
import com.cftech.sys.security.UserUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.util.List;


/**
 * 物流信息Controller
 * <p>
 * 权限字符串说明:
 * 查看:public static final String LOGISTICS_VIEW = "qy:logistics:view"
 * 查看:public static final String LOGISTICS_EDIT = "qy:logistics:edit"
 *
 * @author Buyj
 * @date: 2020-11-29 17:19
 */
@Slf4j
@Controller
@RequestMapping("/a/logistics")
public class LogisticsController {

    public static final String LOGISTICS_VIEW = "qy:logistics:view";
    public static final String LOGISTICS_EDIT = "qy:logistics:edit";

    @Autowired
    private LogisticsService logisticsService;


    //列表页面
    @RequiresPermissions(value = LOGISTICS_VIEW)
    @RequestMapping("/list")
    public String list(HttpServletRequest request, Model model, String mailNo) {
        Long accountId = UserUtils.getmpaccounts(request);
        model.addAttribute("accountId", accountId);
        model.addAttribute("mailNo", mailNo);
        return "logistics/logisticslist";
    }

    //编辑页面（新增、修改）
    @RequiresPermissions(value = LOGISTICS_VIEW)
    @RequestMapping("/form")
    public String form(HttpServletRequest request, String id, Model model) {
        if (!StringUtils.isEmpty(id)) {
            Logistics logistics = logisticsService.fetchById(id);
            model.addAttribute("data", logistics);

            Conds conds = new Conds();
            conds.equal("del_flag", Constants.DEL_FLAG_0);
            conds.equal("mail_no", logistics.getMailNo());
            Sort sort = new Sort( "create_time", OrderType.DESC);
            List<Logistics> routes = logisticsService.fetchSearchByPage(conds, sort, 0, 0);
            model.addAttribute("routes", routes);

        }
        return "logistics/logisticsform";
    }

    //提交数据（新增、修改）
    @RequiresPermissions(value = LOGISTICS_EDIT)
    @RequestMapping("/formData")
    @ResponseBody
    public JSONObject formData(Logistics logistics, Model model, HttpServletRequest request) {
        Long accountsId = UserUtils.getmpaccounts(request);
        JSONObject rtnJson = new JSONObject();
        try {
            if (logistics != null && logistics.getId() != null) {
                logistics.setUpdateBy(UserUtils.getUser().getId());
                logisticsService.update(logistics);
                rtnJson.put("errorNo", 0);
            } else {
                logistics.setAccountsId(accountsId);
                logistics.setDelFlag(false);
                logistics.setAccountsId(UserUtils.getmpaccounts(request));
                logistics.setCreateBy(UserUtils.getUser().getId());
                logistics.setUpdateBy(UserUtils.getUser().getId());
                logisticsService.save(logistics);
                rtnJson.put("errorNo", 2);
            }
        } catch (Exception e) {
            rtnJson.put("errorNo", 1);
        }
        return rtnJson;
    }

    //获取列表数据
    @RequiresPermissions(value = LOGISTICS_VIEW)
    @RequestMapping(value = "/listData")
    @ResponseBody
    public JSONObject listData(int iDisplayStart, int iDisplayLength, Logistics logistics, HttpServletRequest request) {
        Long accountsId = UserUtils.getmpaccounts(request);
        Conds conds = new Conds();
        conds.equal("t.del_flag", Constants.DEL_FLAG_0);
        conds.equal("t.accounts_id", accountsId);
        if(!StringUtils.isEmpty(logistics.getMailNo())){
            conds.equal("t.mail_no",logistics.getMailNo());
        }
        if (!StringUtils.isEmpty(logistics.getOrderNo())) {
            conds.equal("t.order_no", logistics.getOrderNo());
        }
        Sort sort = new Sort("t.create_time", OrderType.DESC);

        List<Logistics> list = logisticsService.fetchSearchByList(conds, sort, iDisplayStart, iDisplayLength);
        Integer counts = logisticsService.countList(conds);
        JSONObject rtnJson = new JSONObject();
        rtnJson.put("iTotalRecords", counts);
        rtnJson.put("iTotalDisplayRecords", counts);
        rtnJson.put("aaData", list);
        return rtnJson;
    }

    //删除数据
    @RequiresPermissions(value = LOGISTICS_EDIT)
    @RequestMapping("/delete")
    @ResponseBody
    public JSONObject delete(String id) {
        JSONObject rtnJosn = new JSONObject();
        try {
            logisticsService.delete(id);
            rtnJosn.put("errorNo", 0);
        } catch (Exception e) {
            rtnJosn.put("errorNo", 1);
        }
        return rtnJosn;
    }

    @RequestMapping("/exportExcel")
    @RequiresPermissions(value = LOGISTICS_VIEW)
    public void exportExcel(HttpServletRequest request, HttpServletResponse response) {
        Long accountId = UserUtils.getmpaccounts(request);
        Sort sort = new Sort("create_time", OrderType.ASC);
        Conds conds = new Conds();
        conds.equal("del_flag", 0);
        conds.equal("accounts_id", accountId);
        List<Logistics> list = logisticsService.fetchSearchByPage(conds, sort, 0, 0);
        ExcelKit.$Export(Logistics.class, response).toExcel(list, "物流信息信息");
    }

    @RequestMapping("/templateExcel")
    @RequiresPermissions(value = LOGISTICS_VIEW)
    public void templateExcel(HttpServletRequest request, HttpServletResponse response) {
        ExcelKit.$Export(Logistics.class, response).toExcel(null, "物流信息信息");
    }

    @RequestMapping("/importExcel")
    @RequiresPermissions(value = LOGISTICS_EDIT)
    public String importExcel(HttpServletRequest request, MultipartFile file, Model model) {
        Long accountId = UserUtils.getmpaccounts(request);
        if (file == null) {
            return list(request, model, null);
        }
// 构造临时路径来存储上传的文件
        String uploadPath = System.getProperty("java.io.tmpdir");
        File uploadDir = new File(uploadPath);
        if (!uploadDir.exists()) {
            uploadDir.mkdir();
        }
        String fileName = file.getOriginalFilename();
        String filePath = uploadPath + File.separator + fileName;
        File storeFile = new File(filePath);
        try {
            file.transferTo(storeFile);
            ExcelKit.$Import().setEmptyCellValue("").readExcel(storeFile, rowData -> {
                if (!StringUtils.isEmpty(rowData.get(0))) {
                    Logistics logistics = new Logistics();
                    logistics.setAccountsId(accountId);
                    logisticsService.save(logistics);
                }
            });
        } catch (IOException e) {
            log.error(e.getMessage());
        }

        return list(request, model, null);
    }

}
