package com.cftech.base.org.api;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.cftech.core.constants.WxApiConstants;
import com.cftech.core.util.OKHttpUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

/**
 * Created by liuling on 2017/1/8.
 */
public class WxApiUtils {

    private static final Logger logger = LoggerFactory.getLogger(WxApiUtils.class);

    /**
     * 根据code获取人员的信息
     *
     * @param token 企业号的token
     * @param code  微信返回的code
     * @return
     */
    public static JSONObject getUserInfo(String token, String code) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.GET_USER_INFO.replace("ACCESS_TOKEN", token).replace("CODE", code);

        try {
            String retStr = OKHttpUtils.getJSON(url);
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("getUserInfo from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }


    /**
     * 创建部门
     *
     * @param token   企业号的token
     * @param postObj post的对象
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject departmentCreate(String token, JSONObject postObj) {

        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.DEPARTMENT_CREATE.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.postJSON(url, postObj.toJSONString());
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("create department from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 更新部门
     *
     * @param token   企业号的token
     * @param postObj post的对象
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject departmentUpdate(String token, JSONObject postObj) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.DEPARTMENT_UPDATE.replace("ACCESS_TOKEN", token);
        try {
            String retStr = OKHttpUtils.postJSON(url, postObj.toJSONString());
            retObj = JSON.parseObject(retStr);

            return retObj;
        } catch (IOException e) {
            logger.error("update department from weixin got io error:" + e.getMessage());

            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 删除部门
     *
     * @param token 企业号的token
     * @param id    微信中部门的Id
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject departmentDelete(String token, Integer id) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.DEPARTMENT_DELETE.replace("ACCESS_TOKEN", token).replace("ID", "" + id);

        try {
            String retStr = OKHttpUtils.getJSON(url);
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("delete department from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 获取部门列表，包含下级
     *
     * @param token 企业号的token
     * @param id    微信中部门的Id
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject departmentList(String token, Integer id) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.DEPARTMENT_LIST.replace("ACCESS_TOKEN", token).replace("ID", "" + id);

        try {
            String retStr = OKHttpUtils.getJSON(url);
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("list department from weixin got io error:" + e.getMessage());

            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 获取人员
     *
     * @param token  企业号的token
     * @param userid 人员的userid
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject userGet(String token, String userid) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.USER_GET.replace("ACCESS_TOKEN", token).replace("USERID", userid);

        try {
            String retStr = OKHttpUtils.getJSON(url);
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("get user from weixin got io error:" + e.getMessage());
            retObj.put("errcode", "1");
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 创建人员
     *
     * @param token   企业号的token
     * @param postObj post的对象
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject userCreate(String token, JSONObject postObj) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.USER_CREATE.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.postJSON(url, postObj.toJSONString());
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("create user from weixin got io error:" + e.getMessage());
            retObj.put("errcode", "1");
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 更新人员
     *
     * @param token   企业号的token
     * @param postObj post的对象
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject userUpdate(String token, JSONObject postObj) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.USER_UPDATE.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.postJSON(url, postObj.toJSONString());
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("update user from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 删除人员
     *
     * @param token  企业号的token
     * @param userid 需要删除的人员userid
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject userDelete(String token, String userid) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.USER_DELETE.replace("ACCESS_TOKEN", token).replace("USERID", userid);

        try {
            String retStr = OKHttpUtils.getJSON(url);
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("delete user from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 获取人员详情列表
     *
     * @param token      企业号的token
     * @param orgId      获取的微信的部门id
     * @param fetchChild 1/0：是否递归获取子部门下面的成员
     * @param status     0获取全部成员，1获取已关注成员列表，2获取禁用成员列表，4获取未关注成员列表。status可叠加,未填写则默认为4
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject userlist(String token, String orgId, String fetchChild, String status) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.USER_LIST.replace("ACCESS_TOKEN", token)
                .replace("DEPARTMENT_ID", orgId)
                .replace("FETCH_CHILD", fetchChild)
                .replace("STATUS", status);
        try {
            String retStr = OKHttpUtils.getJSON(url);
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("userlist from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 批量删除人员
     *
     * @param token   企业号的token
     * @param postObj post的对象
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject userBatchDelete(String token, JSONObject postObj) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.USER_BATCHDELETE.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.postJSON(url, postObj.toJSONString());
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("batch delete user from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 移动部门下的人员至目标部门中
     *
     * @param token      企业号的token
     * @param from       需要移动的部门
     * @param to         人员移动至的部门
     * @param fetchchild 是否包含自部门
     * @param status     人员的状态，0全部成员，1已关注，2禁用，4未关注
     * @return 返回的结果，或者错误信息
     */
    public static JSONObject userMoveByDepartid(String token, int from, int to, String fetchchild, String status) {
        JSONObject retObj = new JSONObject();

        JSONArray successIds = new JSONArray();
        JSONArray failureIds = new JSONArray();

        String url = WxApiConstants.USER_UPDATE.replace("ACCESS_TOKEN", token);
        String userListUrl = WxApiConstants.USER_SIMPLELIST.replace("ACCESS_TOKEN", token)
                .replace("DEPARTMENT_ID", "" + from).replace("FETCH_CHILD", fetchchild).replace("STATUS", status);
        try {
            JSONObject postObj = new JSONObject();
            JSONArray departments = new JSONArray();
            departments.add(to);
            postObj.put("department", departments);

            // 获取部门下的人员列表
            String userList = OKHttpUtils.getJSON(userListUrl);
            logger.info("userList= {}", userList);
            JSONObject userListObj = JSONObject.parseObject(userList);
            if (userListObj.getIntValue("errcode") == 0) {
                JSONArray userListArr = userListObj.getJSONArray("userlist");
                int userSize = userListArr.size();
                logger.info("get departmentid = {}, usersize = {} ", from, userSize);
                for (int i = 0; i < userSize; i++) {
                    JSONObject userObj = userListArr.getJSONObject(i);
                    String userid = userObj.getString("userid");
                    postObj.put("userid", userid);
                    JSONObject rObj = userUpdate(token, postObj);

                    String returnStr = rObj.toJSONString();
                    if (rObj.getIntValue("errcode") == 0) {
                        successIds.add(userid);
                    } else {
                        failureIds.add(userid);
                    }
                }
                retObj.put("errcode", 0);
                retObj.put("successids", successIds);
                retObj.put("failureids", failureIds);
                return retObj;
            } else {
                retObj.put("errcode", 1);
                retObj.put("errmsg", "get userlist failed");
                return retObj;
            }
        } catch (IOException e) {
            logger.error("userMoveByDepartid from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }

    }


    /**
     * 创建标签
     *
     * @param token   企业号的token
     * @param postObj post的对象
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject tagCreate(String token, JSONObject postObj) {

        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.TAG_CREATE.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.postJSON(url, postObj.toJSONString());
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("create tag from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 更新标签
     *
     * @param token   企业号的token
     * @param postObj post的对象
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject tagUpdate(String token, JSONObject postObj) {

        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.TAG_UPDATE.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.postJSON(url, postObj.toJSONString());
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("update tag from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 删除标签
     *
     * @param token 企业号的token
     * @param tagid 需要删除的标签Id
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject tagDelete(String token, String tagid) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.TAG_DELETE.replace("ACCESS_TOKEN", token).replace("TAGID", tagid);

        try {
            String retStr = OKHttpUtils.getJSON(url);
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("delete tag from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 增加标签成员
     *
     * @param token   企业号的token
     * @param postObj post的对象
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject tagAddUsers(String token, JSONObject postObj) {

        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.TAG_ADDUSERS.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.postJSON(url, postObj.toJSONString());
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("tag tagAddUsers from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 删除标签成员
     *
     * @param token   企业号的token
     * @param postObj post的对象
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject tagDeleteUsers(String token, JSONObject postObj) {

        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.TAG_DELETEUSERS.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.postJSON(url, postObj.toJSONString());
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("tag tagDeleteUsers from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 获取标签列表
     *
     * @param token 企业号的token
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject tagList(String token) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.TAG_LIST.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.getJSON(url);
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("list tag from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 获取标签成员
     *
     * @param token 企业号的token
     * @param tagid 需要获取成员的标签Id
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject tagUserList(String token, String tagid) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.TAG_USER_LIST
                .replace("ACCESS_TOKEN", token)
                .replace("TAGID", tagid);

        try {
            String retStr = OKHttpUtils.getJSON(url);
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("tag user list from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 发送消息的接口
     *
     * @param token   企业号的token
     * @param postObj 发送的消息内容体
     * @return {
     * "errcode": 0,
     * "errmsg": "ok",
     * "invaliduser": "UserID1",
     * "invalidparty":"PartyID1",
     * "invalidtag":"TagID1"
     * }
     */
    public static JSONObject sendMessage(String token, JSONObject postObj) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.SEND_MESSAGE.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.postJSON(url, postObj.toJSONString());
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("tag tagDeleteUsers from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

    /**
     * 获取企业号应用概况列表
     *
     * @param token 企业号的token
     * @return 微信企业号接口返回的结果，或者错误信息
     */
    public static JSONObject agentList(String token) {
        JSONObject retObj = new JSONObject();
        String url = WxApiConstants.AGENT_GET.replace("ACCESS_TOKEN", token);

        try {
            String retStr = OKHttpUtils.getJSON(url);
            retObj = JSON.parseObject(retStr);
            return retObj;
        } catch (IOException e) {
            logger.error("get agent from weixin got io error:" + e.getMessage());
            retObj.put("errcode", 1);
            retObj.put("errmsg", e.getMessage());
            return retObj;
        }
    }

}
