package com.cftech.addresst.web;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.support.spring.GenericFastJsonRedisSerializer;
import com.cftech.addresst.model.Address;
import com.cftech.addresst.model.Area;
import com.cftech.addresst.service.AddressService;
import com.cftech.core.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.io.Serializable;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author ：licc
 * @date ：Created in 2020/10/19 18:11
 * @description：收货地址
 */
@Slf4j
@RestController
@CrossOrigin
@RequestMapping("mobile/auth/address")
public class MobileAddressController {

    @Autowired
    private AddressService addressService;

    @Autowired
    private RedisTemplate<String, String> redisTemplate;

    private RedisTemplate<String, String> getConfig() {
        GenericFastJsonRedisSerializer genericFastJsonRedisSerializer = new GenericFastJsonRedisSerializer();
        //StringRedisSerializer stringSerializer = new StringRedisSerializer();
        redisTemplate.setKeySerializer(genericFastJsonRedisSerializer);
        redisTemplate.setHashKeySerializer(genericFastJsonRedisSerializer);
        redisTemplate.setHashValueSerializer(genericFastJsonRedisSerializer);
        return redisTemplate;
    }

    /**
     * @return
     * @Description 个人收货地址数据回填
     * @Date 9:53 2020/10/20
     * @Param
     **/
    @RequestMapping(value = "/addressList", method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public JSONObject addressList(String openId, String appId, Long addressId) {
        return addressService.addressList(openId, appId, addressId);
    }

    /**
     * @return
     * @Description 根据id查询收货地址
     * @Date 9:53 2020/10/20
     * @Param
     **/
    @RequestMapping(value = "/addressId", method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public JSONObject addressId(Long id,String appId) {
        return addressService.addressId(id,appId);
    }

    /**
     * 增加redis缓存
     * @return
     * @Description 省市区信息回填
     * @Date 9:53 2020/10/20
     * @Param
     **/
    @RequestMapping(value = "/listArea", method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public JSONObject listArea(String appId) {
        JSONObject rtnJson = new JSONObject();
        try {
            String key = "AIDEA_AREA_LIST_" + appId;
            if (getConfig().hasKey(key)) {
                String listArea = getConfig().opsForValue().get(key);
                if (StringUtils.isNoneBlank(listArea)) {
                    JSONArray arr = JSON.parseArray(listArea);
                    rtnJson.put("errorNo", "0");
                    rtnJson.put("data", arr);
                    return rtnJson;
                }
            }
            List<Area> areas = addressService.listArea();
            if (areas != null && areas.size() != 0) {
                //缓存至redis
                getConfig().opsForValue().set(key, JSON.toJSONString(areas));
                getConfig().expire(key, 24, TimeUnit.HOURS);

                rtnJson.put("errorNo", "0");
                rtnJson.put("data", areas);
                return rtnJson;
            }
        } catch (Exception e) {
            e.printStackTrace();
            rtnJson.put("errorNo", "1");
        }
        rtnJson.put("errorNo", "1");
        return rtnJson;
    }

    /**
     * @return
     * @Description 个人收货地址新增
     * @Date 9:53 2020/10/20
     * @Param
     **/
    @RequestMapping(value = "/addressAdd", method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public JSONObject addressAdd(Address address) {
        return addressService.addressAdd(address);
    }

    /**
     * @return
     * @Description 收货地址修改
     * @Date 15:19 2020/10/20
     * @Param
     **/
    @RequestMapping(value = "/updateAddress", method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public JSONObject updateAddress(Address address) {
        return addressService.updateAddress(address);
    }

    /**
     * @return
     * @Author Licc
     * @Description 收货地址删除
     * @Date 15:52 2020/10/20
     * @Param
     **/
    @RequestMapping(value = "/deleteAddress", method = {RequestMethod.GET, RequestMethod.POST}, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public JSONObject deleteAddress(String id,String appId) {
        return addressService.deleteAddress(id,appId);
    }


}
